//==============================================================================
// Project:		Wealth Transfers and their Economic Effects
// File name:	Bequests and gifts received
// Objective: 	Generate summary statistics for inheritances and gifts received.
//
// Created: 	16/04/2021
// Modified: 	15/11/2021
//==============================================================================

////////////////////
/// Inheritances ///
////////////////////

*************************************
*** Loading and cleaning the data ***
*************************************

clear

cd "XXXX" 	// set content directory here

use HILDA_restricted_combined_rescaled	// importing the unconfidentialised HILDA data for waves 1-19 that includes the new weights for residents of non-private dwellings

drop if hgint == 0	// dropping non-responding persons

replace oifinha = . if oifinha <= 0	// replacing negative values for bequests as missing
replace oifinh = . if oifinh != 1	// replacing zero values for receiving a bequest as missing

*************************************
*** Generating summary statistics ***
*************************************

by wave, sort: egen count_bequest = count(oifinh)	// calculating the number of bequests received each year

by wave, sort: egen mean_bequest = mean(oifinha)	// calculating the mean value of bequests received each year

by wave, sort: egen median_bequest = median(oifinha)	// calculating the median value of bequests received each year

**********************************************
*** Generating weighted summary statistics ***
**********************************************

gen pop_oifinha = oifinha * benchmarked_new_hhwtrp	// multiplying bequests received by the responding-person population weights

by wave, sort: egen sum_bequest = sum(pop_oifinha)	// calculating the aggregate value of bequests received each year

by wave, sort: egen mean_rp_wt = mean(benchmarked_new_hhwtrp)	// calculating the mean benchmarked responding-person population weight

gen wt_oifinh_pop = oifinh * benchmarked_new_hhwtrp	// multiplying the bequest indicator variable by the responding-person population weights

gen wt_oifinha = oifinha *  benchmarked_new_hhwtrp / mean_rp_wt	// multiplying the value of bequests received by the responding-person weights

by wave, sort: egen wt_count_bequest_pop = sum(wt_oifinh_pop)	// calculating the number of bequests received by the population each year after the weights have been applied

* calculating the median value of inheritances received each year after the weights have been applied
gen wt_median_inheritance = .
foreach x of numlist 1(1)19 {
	sum oifinha [w = benchmarked_new_hhwtrp] if wave == `x', detail
	gen median_inheritance_`x' = r(p50)
	replace wt_median_inheritance = median_inheritance_`x' if wave == `x'
	drop median_inheritance_`x'
}

gen wt_count_bequest = wt_count_bequest_pop / mean_rp_wt	// calculating the number of bequests recieved each year after the weights have been applied

gen wt_mean_bequest = sum_bequest / wt_count_bequest_pop	// calculating the mean value of bequests received each year after the weights have been applied

by wave, sort: egen wt_median_bequest = median(wt_oifinha)	// calculating the median value of bequests received each year after the weights have been applied

collapse(mean) count_bequest mean_bequest median_bequest sum_bequest wt_count_bequest wt_count_bequest_pop wt_mean_bequest wt_median_bequest, by(wave)	// collapsing the dataset and keeping the key summary statistics for each year

save "Bequests Received.dta", replace	// saving the dataset

/////////////
/// Gifts ///
/////////////

*************************************
*** Loading and cleaning the data ***
*************************************

clear

cd "XXXX"	// set content directory here

use HILDA_restricted_combined_rescaled	// importing the unconfidentialised HILDA data for waves 1-19 that includes the new weights for residents of non-private dwellings

drop if hgint == 0	// dropping non-responding persons

replace oifpnta = . if oifpnta <= 0	// replacing negative values for parental transfers as missing
replace oifpnt = . if oifpnt != 1	// replacing as missing those who did not receive a parental transfer

replace oifohha = . if oifohha <= 0	// replacing negative values for other transfers as missing
replace oifohh = . if oifohh != 1	// replacing as missing those who did not receive an other transfer

*************************************
*** Generating summary statistics ***
*************************************

* parental gifts
by wave, sort: egen count_gift = count(oifpnt)	// calculating the number of individuals who receieved a parental transfer each year

by wave, sort: egen mean_gift = mean(oifpnta)	// calculating the mean value of parental transfers receieved each year

by wave, sort: egen median_gift = median(oifpnta)	// calculating the median value of parental transfers received each year


* other gifts
by wave, sort: egen count_other_gift = count(oifohh)	// calculating the number of individuals who receieved an other transfer each year

by wave, sort: egen mean_other_gift = mean(oifohha)	// calculating the mean value of other transfers receieved each year

by wave, sort: egen median_other_gift = median(oifohha)	// calculating the median value of other transfers received each year

**********************************************
*** Generating weighted summary statistics ***
**********************************************

* parental gifts
gen pop_oifpnta = oifpnta * benchmarked_new_hhwtrp	// multiplying parental transfers by the responding-person population weights

by wave, sort: egen sum_gift = sum(pop_oifpnta)	// calculating the aggregate value of parental transfers received each year

by wave, sort: egen mean_rp_wt = mean(benchmarked_new_hhwtrp)

gen wt_pop_oifpnt = oifpnt * benchmarked_new_hhwtrp	// multiplying the parental transfer indicator variable by the responding-person population weights

gen wt_oifpnta = oifpnta * benchmarked_new_hhwtrp / mean_rp_wt	// multiplying parental transfers by the responding-person weights

by wave, sort: egen wt_count_gift_pop = sum(wt_pop_oifpnt)	// calculating the number of individuals in the population receiving parental transfers each year after the weights have been applied

gen wt_count_gift = wt_count_gift_pop / mean_rp_wt	// calculating the number of gifts received each year after the weights have been applied

gen wt_mean_gift = sum_gift / wt_count_gift_pop	// calculating the mean value of gifts received each year after the weights have been applied

* calculating the median value of parental gifts received each year after the weights have been applied
gen wt_median_gift = .
foreach x of numlist 1(1)19 {
	sum oifpnta [w = benchmarked_new_hhwtrp] if wave == `x', detail
	gen median_gift_`x' = r(p50)
	replace wt_median_gift = median_gift_`x' if wave == `x'
	drop median_gift_`x'
}

* other gifts
gen pop_oifohha = oifohha * benchmarked_new_hhwtrp	// multiplying parental transfers by the responding-person population weights

by wave, sort: egen sum_other_gift = sum(pop_oifohha)	// calculating the aggregate value of parental transfers received each year

gen wt_pop_oifohh = oifohh * benchmarked_new_hhwtrp	// multiplying the parental transfer indicator variable by the responding-person population weights

gen wt_oifohha = oifohha * benchmarked_new_hhwtrp / mean_rp_wt	// multiplying parental transfers by the responding-person weights

by wave, sort: egen wt_count_other_gift_pop = sum(wt_pop_oifohh)	// calculating the number of individuals in the population receiving parental transfers each year after the weights have been applied

gen wt_count_other_gift = wt_count_other_gift_pop / mean_rp_wt	// calculating the number of gifts received each year after the weights have been applied

gen wt_mean_other_gift = sum_other_gift / wt_count_other_gift_pop	// calculating the mean value of gifts received each year after the weights have been applied

* calculating the median value of other gifts received each year after the weights have been applied
gen wt_median_other_gift = .
foreach x of numlist 1(1)19 {
	sum oifpnta [w = benchmarked_new_hhwtrp] if wave == `x', detail
	gen median_other_gift_`x' = r(p50)
	replace wt_median_other_gift = median_other_gift_`x' if wave == `x'
	drop median_other_gift_`x'
}

collapse(mean) count_* mean_* median_* sum_* wt_count_* wt_mean_* wt_median_*, by(wave)	// collapsing the dataset, keeping the key summary statistics for each year

* all gifts
gen count_all_gift = count_gift + count_other_gift	// number of transfers reported

gen wt_count_all_gift = wt_count_gift + wt_count_other_gift	// weighted number of transfers

gen wt_count_all_gift_pop = wt_count_gift_pop + wt_count_other_gift_pop	// population-weighted number of transfers

gen sum_all_gift = sum_gift + sum_other_gift	// sum of all transfers

save "Gifts Received.dta", replace	// saving the dataset

*******************************************
*** Merging the bequests and gifts data ***
*******************************************

merge 1:1 wave using "Bequests Received.dta", nogen	// merging the gifts data with the bequests data

gen sum_transfers = sum_all_gift + sum_bequest	// calculating the total amount of wealth transferred through either gifts or bequests each year
gen count_transfers = count_all_gift + count_bequest
gen wt_count_transfers = wt_count_all_gift + wt_count_bequest
gen wt_count_transfers_pop = wt_count_all_gift_pop + wt_count_bequest_pop

*******************************************
*** Converting to constant 2018 dollars ***
*******************************************

gen year = wave + 2000	// generating a year variable for merging
merge 1:1 year using CPI_FY_2018.dta	// merging in CPI data
drop if _merge == 2	// dropping not-matched observations
drop _merge wave sum_nh_wte sum_benchmarked_hhwte	// dropping unused variables

local dollar_vars mean_bequest mean_gift mean_other_gift median_bequest median_gift median_other_gift sum_bequest sum_gift sum_other_gift sum_all_gift sum_transfers wt_mean_bequest wt_mean_gift wt_mean_other_gift wt_median_bequest wt_median_gift wt_median_other_gift	// creating a local macro including all dollar variables

* generating real variables in constant 2018 dollars
foreach x in `dollar_vars' {
    gen real_`x' = `x' * (100 / cpi)
}

*************************************
*** Exporting and saving the data ***
*************************************

keep year count_bequest wt_count_bequest_pop real_wt_mean_bequest real_wt_median_bequest real_sum_bequest count_gift wt_count_gift_pop real_wt_mean_gift real_wt_median_gift real_sum_gift count_other_gift wt_count_other_gift_pop real_sum_other_gift count_all_gift wt_count_all_gift_pop real_sum_all_gift count_transfers wt_count_transfers_pop real_sum_transfers cpi

order year count_bequest wt_count_bequest_pop real_wt_mean_bequest real_wt_median_bequest real_sum_bequest count_gift wt_count_gift_pop real_wt_mean_gift real_wt_median_gift real_sum_gift count_other_gift wt_count_other_gift_pop real_sum_other_gift count_all_gift wt_count_all_gift_pop real_sum_all_gift count_transfers wt_count_transfers_pop real_sum_transfers cpi

save "Bequests and Gifts Received", replace	// saving the combined dataset